<?php

namespace App\Http\Controllers\User;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\SocialConnect;
use App\Models\Platform;
use App\Models\SocialAccount;
use Illuminate\Http\Request;

class SocialAccountController extends Controller {
    public function index() {
        $pageTitle = "Social Accounts";
        $platforms = Platform::active()->get();
        $accounts  = SocialAccount::with('platform')->where('user_id', auth()->id())->get();
        return view('Template::user.social_account', compact('pageTitle', 'platforms', 'accounts'));
    }

    public function connect($provider) {
        if (!activePlan()) {
            $notify[] = ['error', 'You need to subscribe to a plan to connect social accounts'];
            return to_route('user.member.index')->withNotify($notify);
        }

        $user = activePlan();

        if ($user->total_channel <= $user->connected_channel) {
            $notify[] = ['error', 'You have reached your channel limit. Please upgrade your plan'];
            return to_route('user.member.index')->withNotify($notify);
        }

        $socialConnect = new SocialConnect($provider);
        return $socialConnect->redirectDriver();
    }

    public function callback($provider) {
        $socialLogin = new SocialConnect($provider);
        try {
            return $socialLogin->login();
        } catch (\Exception $e) {
            $notify[] = ['error', $e->getMessage()];
            return to_route('user.social.index')->withNotify($notify);
        }
    }

    public function connectTelegram(Request $request) {
        $request->validate([
            'channel_id' => 'required',
            'bot_id'     => 'required',
        ]);

        $user      = auth()->user();
        $channelId = $request->channel_id;
        $botId     = $request->bot_id;

        $platform = Platform::active()->where('name', 'Telegram')->first();
        $account  = SocialAccount::where('user_id', $user->id)->where('platform_id', $platform->id)->where('profile_id', $channelId)->exists();

        if ($account) {
            $notify[] = ['error', 'Social account already connected'];
            return back()->withNotify($notify);
        }
        if (!$platform) {
            $notify[] = ['error', 'Telegram platform not found'];
            return back()->withNotify($notify);
        }

        try {
            $infoEndpoint = "https://api.telegram.org/bot{$botId}/getChat?chat_id={$channelId}";
            $channelInfo  = json_decode(file_get_contents($infoEndpoint));

            $photoId       = $channelInfo?->result?->photo?->big_file_id ?? '';
            $photoEndpoint = "https://api.telegram.org/bot{$botId}/getFile?file_id={$photoId}";
            $photoInfo     = json_decode(file_get_contents($photoEndpoint));

            $photoUrl = "https://api.telegram.org/file/bot{$botId}/{$photoInfo?->result?->file_path}";

            $socialAccount                = new SocialAccount();
            $socialAccount->user_id       = $user->id;
            $socialAccount->platform_id   = $platform->id;
            $socialAccount->profile_id    = $channelId;
            $socialAccount->profile_name  = $channelInfo?->result?->title ?? '';
            $socialAccount->profile_image = $photoUrl;
            $socialAccount->access_token  = $botId;
            $socialAccount->expires_at    = now()->addYear();
            $socialAccount->status        = Status::ENABLE;
            $socialAccount->save();

            $user->connected_channel += 1;
            $user->save();

            $notify[] = ['success', 'Telegram account connected successfully'];
            return back()->withNotify($notify);
        } catch (\Throwable $th) {
            $notify[] = ['error', $th->getMessage()];
            return back()->withNotify($notify);
        }
    }
}
