<?php

namespace App\Http\Controllers\Gateway;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\FormProcessor;
use App\Lib\Referral;
use App\Models\AdminNotification;
use App\Models\AppliedCoupon;
use App\Models\Coupon;
use App\Models\Deposit;
use App\Models\GatewayCurrency;
use App\Models\Plan;
use App\Models\Subscription;
use App\Models\Transaction;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class PaymentController extends Controller {
    public function deposit() {
        $gatewayCurrency = GatewayCurrency::whereHas('method', function ($gate) {
            $gate->where('status', Status::ENABLE);
        })->with('method')->orderby('name')->get();

        $subscriptionId = session('subscription_id');

        if (!$subscriptionId) {
            $notify[] = ['error', 'Oops! Session invalid'];
            return redirect()->route('user.home')->withNotify($notify);
        }

        $subscription = Subscription::where('status', Status::NO)->where('id', $subscriptionId)->first();
        if (!$subscription) {
            $notify[] = ['error', 'Oops! Subscription not found'];
            return redirect()->route('user.home')->withNotify($notify);
        }
        $amount    = $subscription->price;
        $pageTitle = 'Payment Methods';
        $plan      = $subscription->plan;

        return view('Template::user.payment.deposit', compact('gatewayCurrency', 'pageTitle', 'amount', 'plan', 'subscription'));
    }

    public function depositInsert(Request $request) {
        $request->validate([
            'amount'   => 'required|numeric|gt:0',
            'gateway'  => 'required',
            'currency' => 'required',
        ]);

        $user = auth()->user();

        $subscriptionId = session()->get('subscription_id');
        if (!$subscriptionId) {
            $notify[] = ['error', 'Oops! Session invalid'];
            return back()->withNotify($notify);
        }

        $subscription = Subscription::inactive()->where('id', $subscriptionId)->first();
        if (!$subscription) {
            $notify[] = ['error', 'Oops! Subscription not found'];
            return back()->withNotify($notify);
        }
        $amount = $subscription->price;

        if (session()->has('coupon')) {
            $couponCode = session()->get('coupon')['code'];
            $coupon     = Coupon::active()->where('code', $couponCode)->first();
            if (!$coupon) {
                session()->forget('coupon');
                $notify[] = ['error', 'Invalid coupon'];
                return back()->withNotify($notify);
            }

            $couponValidation = couponValidation($coupon, $amount);
            if ($couponValidation['status'] == 'error') {
                $notify[] = ['error', $couponValidation['message']];
            }

            $amount = $coupon->discountAmount($request->amount);

            $subscription->coupon_id       = $coupon->id;
            $subscription->discount_amount = $amount;
            $subscription->save();

            session()->forget('coupon');
        }

        if ($request->gateway == "balance_payment") {
            if ($user->balance < $amount) {
                $notify[] = ['error', 'Insufficient balance'];
                return back()->withNotify($notify);
            }

            self::updateSubscription($user, $subscription->id, $amount);

            $user->balance -= $amount;
            $user->save();

            $data                  = new Deposit();
            $data->user_id         = $user->id;
            $data->subscription_id = $subscription->id;
            $data->method_code     = 0;
            $data->method_currency = gs('currency');
            $data->amount          = $amount;
            $data->charge          = 0;
            $data->rate            = 0;
            $data->final_amount    = $amount;
            $data->btc_amount      = 0;
            $data->btc_wallet      = "";
            $data->trx             = getTrx();
            $data->status          = Status::PAYMENT_SUCCESS;
            $data->success_url     = route('user.deposit.history');
            $data->failed_url      = route('user.deposit.history');
            $data->save();

            $transaction               = new Transaction();
            $transaction->user_id      = $user->id;
            $transaction->amount       = $amount;
            $transaction->post_balance = $user->balance;
            $transaction->charge       = 0;
            $transaction->trx          = $data->trx;
            $transaction->details      = 'Plan purchased using balance';
            $transaction->trx_type     = '-';
            $transaction->remark       = 'plan_purchase';
            $transaction->save();

            $notify[] = ['success', 'Plan purchased successfully'];
            return to_route('user.deposit.history')->withNotify($notify);
        }

        $gate = GatewayCurrency::whereHas('method', function ($gate) {
            $gate->where('status', Status::ENABLE);
        })->where('method_code', $request->gateway)->where('currency', $request->currency)->first();

        if (!$gate) {
            $notify[] = ['error', 'Invalid gateway'];
            return back()->withNotify($notify);
        }

        if ($gate->min_amount > $amount || $gate->max_amount < $amount) {
            $notify[] = ['error', 'Please follow deposit limit'];
            return back()->withNotify($notify);
        }

        $charge      = $gate->fixed_charge + ($amount * $gate->percent_charge / 100);
        $payable     = $amount + $charge;
        $finalAmount = $payable * $gate->rate;

        $data                  = new Deposit();
        $data->user_id         = $user->id;
        $data->subscription_id = $subscription->id;
        $data->method_code     = $gate->method_code;
        $data->method_currency = strtoupper($gate->currency);
        $data->amount          = $amount;
        $data->charge          = $charge;
        $data->rate            = $gate->rate;
        $data->final_amount    = $finalAmount;
        $data->btc_amount      = 0;
        $data->btc_wallet      = "";
        $data->trx             = getTrx();
        $data->success_url     = route('user.deposit.history');
        $data->failed_url      = route('user.deposit.history');
        $data->save();
        session()->put('Track', $data->trx);
        return to_route('user.deposit.confirm');
    }

    public function depositConfirm() {
        $track   = session()->get('Track');
        $deposit = Deposit::where('trx', $track)->where('status', Status::PAYMENT_INITIATE)->orderBy('id', 'DESC')->with('gateway')->firstOrFail();

        if ($deposit->method_code >= 1000) {
            return to_route('user.deposit.manual.confirm');
        }

        $dirName = $deposit->gateway->alias;
        $new     = __NAMESPACE__ . '\\' . $dirName . '\\ProcessController';

        $data = $new::process($deposit);
        $data = json_decode($data);

        if (isset($data->error)) {
            $notify[] = ['error', $data->message];
            return back()->withNotify($notify);
        }
        if (isset($data->redirect)) {
            return redirect($data->redirect_url);
        }

        // for Stripe V3
        if (isset($data->session)) {
            $deposit->btc_wallet = $data->session->id;
            $deposit->save();
        }

        $pageTitle = 'Payment Confirm';
        return view("Template::$data->view", compact('data', 'pageTitle', 'deposit'));
    }

    private static function updateSubscription($user, $subscriptionId, $amount) {
        $existsSubscription = Subscription::where('user_id', $user->id)->where('status', Status::ENABLE)->first();
        if ($existsSubscription) {
            $existsSubscription->status = Status::DISABLE;
            $existsSubscription->save();
        }

        $subscription         = Subscription::inactive()->where('id', $subscriptionId)->first();
        $subscription->status = Status::PAYMENT_SUCCESS;
        $subscription->save();

        if ($subscription->coupon_id) {
            $coupon                   = Coupon::find($subscription->coupon_id);
            $appliedCoupon            = new AppliedCoupon();
            $appliedCoupon->user_id   = $user->id;
            $appliedCoupon->coupon_id = $coupon->id;
            $appliedCoupon->amount    = $amount;
            $appliedCoupon->save();
        }

        $plan = $subscription->plan;

        if ($plan) {
            $user->plan_id           = $subscription->plan_id;
            $user->total_channel     = $subscription->total_channel;
            $user->total_schedule    = $subscription->total_schedule;
            $user->expired_at        = $subscription->expired_date;
            $user->connected_channel = 0;
            $user->used_schedule     = 0;
            $user->save();

            if ($user->socialAccounts->isNotEmpty()) {
                $user->socialAccounts->each(function ($account) {
                    $account->delete();
                });
            }

            notify($user, 'SUBSCRIBE_PLAN', [
                'plan'     => $plan->name,
                'price'    => showAmount($subscription->price),
                'duration' => $subscription->expired_date,
            ]);
        }
    }

    public static function userDataUpdate($deposit, $isManual = null) {
        if ($deposit->status == Status::PAYMENT_INITIATE || $deposit->status == Status::PAYMENT_PENDING) {
            $deposit->status = Status::PAYMENT_SUCCESS;
            $deposit->save();

            $user = User::find($deposit->user_id);
            self::updateSubscription($user, $deposit->subscription_id, $deposit->amount);
            $methodName = $deposit->methodName();

            if (!$isManual) {
                $adminNotification            = new AdminNotification();
                $adminNotification->user_id   = $user->id;
                $adminNotification->title     = 'Payment successful via ' . $methodName;
                $adminNotification->click_url = urlPath('admin.deposit.successful');
                $adminNotification->save();
            }

            if (gs()->deposit_commission) {
                Referral::levelCommission($user, $deposit->amount, $deposit->trx, 'deposit_commission');
            }

            notify($user, $isManual ? 'DEPOSIT_APPROVE' : 'DEPOSIT_COMPLETE', [
                'method_name'     => $methodName,
                'method_currency' => $deposit->method_currency,
                'method_amount'   => showAmount($deposit->final_amount, currencyFormat: false),
                'amount'          => showAmount($deposit->amount, currencyFormat: false),
                'charge'          => showAmount($deposit->charge, currencyFormat: false),
                'rate'            => showAmount($deposit->rate, currencyFormat: false),
                'trx'             => $deposit->trx,
                'post_balance'    => showAmount($user->balance),
            ]);
        }
    }

    public function manualDepositConfirm() {
        $track = session()->get('Track');
        $data  = Deposit::with('gateway')->where('status', Status::PAYMENT_INITIATE)->where('trx', $track)->first();
        abort_if(!$data, 404);
        if ($data->method_code > 999) {
            $pageTitle = 'Confirm Payment';
            $method    = $data->gatewayCurrency();
            $gateway   = $method->method;
            return view('Template::user.payment.manual', compact('data', 'pageTitle', 'method', 'gateway'));
        }
        abort(404);
    }

    public function manualDepositUpdate(Request $request) {
        $track = session()->get('Track');
        $data  = Deposit::with('gateway')->where('status', Status::PAYMENT_INITIATE)->where('trx', $track)->first();
        abort_if(!$data, 404);
        $gatewayCurrency = $data->gatewayCurrency();
        $gateway         = $gatewayCurrency->method;
        $formData        = $gateway->form->form_data;

        $formProcessor  = new FormProcessor();
        $validationRule = $formProcessor->valueValidation($formData);
        $request->validate($validationRule);
        $userData = $formProcessor->processFormData($request, $formData);

        $data->detail = $userData;
        $data->status = Status::PAYMENT_PENDING;
        $data->save();

        $adminNotification            = new AdminNotification();
        $adminNotification->user_id   = $data->user->id;
        $adminNotification->title     = 'Payment request from ' . $data->user->username;
        $adminNotification->click_url = urlPath('admin.deposit.details', $data->id);
        $adminNotification->save();

        notify($data->user, 'DEPOSIT_REQUEST', [
            'method_name'     => $data->gatewayCurrency()->name,
            'method_currency' => $data->method_currency,
            'method_amount'   => showAmount($data->final_amount, currencyFormat: false),
            'amount'          => showAmount($data->amount, currencyFormat: false),
            'charge'          => showAmount($data->charge, currencyFormat: false),
            'rate'            => showAmount($data->rate, currencyFormat: false),
            'trx'             => $data->trx,
        ]);

        $notify[] = ['success', 'You have payment request has been taken'];
        return to_route('user.deposit.history')->withNotify($notify);
    }

    public function applyCoupon(Request $request) {
        if (session()->has('coupon')) {
            return errorResponse('You have already applied a coupon');
        }

        $validator = Validator::make($request->all(), ['code' => 'required|string']);
        if ($validator->fails()) {
            return errorResponse($validator->errors());
        }

        $coupon = Coupon::active()->where('code', $request->code)->first();
        if (!$coupon) {
            return errorResponse('Invalid coupon code');
        }

        $subscriptionId = session()->get('subscription_id');
        if (!$subscriptionId) {
            $notify[] = ['error', 'Oops! Session invalid'];
            return back()->withNotify($notify);
        }

        $subscription = Subscription::inactive()->where('id', $subscriptionId)->first();
        if (!$subscription) {
            $notify[] = ['error', 'Oops! Subscription not found'];
            return back()->withNotify($notify);
        }

        $amount           = $subscription->price;
        $couponValidation = couponValidation($coupon, $amount);
        if ($couponValidation['status'] == 'error') {
            return errorResponse($couponValidation['message']);
        }

        $discountAmount = $coupon->discountAmount($amount);
        session()->put('coupon', ['code' => $request->code, 'amount' => getAmount($discountAmount), 'couponAmount' => $coupon->amount]);
        return successResponse('Coupon applied successfully', [
            'coupon_code'   => $coupon->coupon_code,
            'payableAmount' => getAmount($discountAmount),
            'couponAmount'  => getAmount($coupon->amount),
        ]);
    }

    public function removeCoupon() {
        session()->forget('coupon');
        return successResponse('Coupon removed successfully');
    }
}
