<?php

namespace App\Http\Controllers;

use App\Constants\Status;
use App\Lib\CurlRequest;
use App\Lib\MetaPublisher;
use App\Models\CronJob;
use App\Models\CronJobLog;
use App\Models\Post;
use App\Models\SocialAccount;
use App\Models\User;
use Carbon\Carbon;

class CronController extends Controller {
    public function cron() {
        $general            = gs();
        $general->last_cron = now();
        $general->save();

        $crons = CronJob::with('schedule');

        if (request()->alias) {
            $crons->where('alias', request()->alias);
        } else {
            $crons->where('next_run', '<', now())->where('is_running', Status::YES);
        }
        $crons = $crons->get();

        foreach ($crons as $cron) {
            $cronLog              = new CronJobLog();
            $cronLog->cron_job_id = $cron->id;
            $cronLog->start_at    = now();
            if ($cron->is_default) {
                $controller = new $cron->action[0];
                try {
                    $method = $cron->action[1];
                    $controller->$method();
                } catch (\Exception $e) {
                    $cronLog->error = $e->getMessage();
                }
            } else {
                try {
                    CurlRequest::curlContent($cron->url);
                } catch (\Exception $e) {
                    $cronLog->error = $e->getMessage();
                }
            }
            $cron->last_run = now();
            $cron->next_run = now()->addSeconds((int) $cron->schedule->interval);
            $cron->save();

            $cronLog->end_at = $cron->last_run;

            $startTime         = Carbon::parse($cronLog->start_at);
            $endTime           = Carbon::parse($cronLog->end_at);
            $diffInSeconds     = $startTime->diffInSeconds($endTime);
            $cronLog->duration = $diffInSeconds;
            $cronLog->save();
        }
        if (request()->target == 'all') {
            $notify[] = ['success', 'Cron executed successfully'];
            return back()->withNotify($notify);
        }
        if (request()->alias) {
            $notify[] = ['success', keyToTitle(request()->alias) . ' executed successfully'];
            return back()->withNotify($notify);
        }
    }

    public function refreshToken() {
        $accounts = SocialAccount::active()->where('refresh_token', '!=', null)->get();
        foreach ($accounts as $account) {
            $metaPublisher = new MetaPublisher($account);
            if ($account->platform_id == Status::YOUTUBE) {
                $metaPublisher->refreshYoutubeAccessToken($account);
            } else if ($account->platform_id == Status::LINKEDIN) {
                $metaPublisher->refreshLinkedinAccessToken($account);
            } else if ($account->platform_id == Status::TWITTER) {
                $metaPublisher->refreshTwitterAccessToken($account);
            }else if ($account->platform_id == Status::TIKTOK) {
                $metaPublisher->refreshTiktokAccessToken($account);
            }
        }
    }

    public function checkSubscription() {
        $users = User::whereNotNull('expired_at')->where('expired_at', '<', now())->limit(20)->get();
        foreach ($users as $user) {
            $user->plan_id           = 0;
            $user->total_channel     = 0;
            $user->total_schedule    = 0;
            $user->connected_channel = 0;
            $user->used_schedule     = 0;
            $user->started_at        = null;
            $user->expired_at        = null;
            $user->save();

            if ($user->socialAccounts->isNotEmpty()) {
                $user->socialAccounts->each(function ($account) {
                    $account->delete();
                });
            }

            userNotification($user->id, 'Your subscription has expired.', route('user.member.index'));
        }
    }

    public function expiredSocialAccount() {
        $accounts = SocialAccount::where('expires_at', '<', now())->get();
        foreach ($accounts as $account) {
            $user                    = $account->user;
            $user->connected_channel = $user->connected_channel > 0 ? $user->connected_channel - 1 : 0;
            $user->save();
            $account->delete();
            userNotification($user->id, 'Your social account has expired.Please re-connect your account.', route('user.social.index'));
        }
    }

    public function postWithSchedule() {

        $postSchedules = Post::where('status', Status::SCHEDULE)
            ->whereDate('schedule_start_date', '<=', today())
            ->whereDate('schedule_end_date', '>=', today())
            ->whereTime('schedule_time', '<=', now()->format('H:i:s'))
            ->where(function ($q) {
                $q->whereNull('last_posted_at')
                    ->orWhereDate('last_posted_at', '<', today());
            })->take(20)->orderBy('schedule_time', 'desc')->get();

        foreach ($postSchedules as $postSchedule) {
            if ($postSchedule->post_type == Status::REMAINDER) {
                notify($postSchedule->user, 'SCHEDULE_REMAINDER', [
                    'user_name '      => $postSchedule->user->username,
                    'site_name '      => gs('site_name'),
                    'post_title '     => strLimit($postSchedule->content, 20),
                    'scheduled_time ' => showDateTime($postSchedule->schedule_time, 'h:i A'),
                ]);

                userNotification($postSchedule->user_id, "Post scheduled successfully", route('user.post.index'));

                $postSchedule->last_posted_at = now();
                $postSchedule->save();

                if ($postSchedule->schedule_end_date == now()->format('Y-m-d')) {
                    $postSchedule->status = Status::PUBLISH;
                    $postSchedule->save();
                }
                continue;
            }

            $publisher = new MetaPublisher($postSchedule);
            $response  = $publisher->publish();

            if ($response['status'] == 'error') {
                userNotification($postSchedule->user_id, $response['message'], route('user.post.index'));
            }

            if ($response['status'] == 'success') {
                $postSchedule->last_posted_at = now();
                $postSchedule->save();

                if (Carbon::parse($postSchedule->schedule_end_date)->toDateString() == today()->toDateString()) {
                    $postSchedule->status = Status::PUBLISH;
                    $postSchedule->save();
                }

                userNotification($postSchedule->user_id, "Post scheduled successfully", route('user.post.index'));
            }
        }
    }
}
